# -*- coding: utf-8 -*-
"""
Daniel Calderon S.

file: botella.py
----------------
Genera una botella como volumen de revolución y lo almacena en un DisplayList.
Utiliza clase Vector3D para modelar la botella.

Control del programa:
1: visualiza o no los ejes de coordenadas
2: visualiza o no como malla de alambre
3: alterna entre smooth y flat
up,down : acerca o aleja la cámara
ESC: terminar
"""

import pygame
from pygame.locals import *

from OpenGL.GL import *
from OpenGL.GLU import *
from OpenGL.GLUT import *

from AuxiliaryFunctions import *
import Vector3D as v3d

from math import pi

#####################################################################

def init_pygame((w,h), title=""):
    pygame.init()
    pygame.display.set_mode((w,h), OPENGL|DOUBLEBUF)
    pygame.display.set_caption(title)

def init_opengl((w,h)):
    reshape((w,h))
    init()
 
def init():
    # setea el color de fondo
    glClearColor(0.0, 0.0, 0.0, 1.0)
	
    # se habilitan las transparencias
    glEnable(GL_BLEND)
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
    # el color debe incluir la cuarta componente, alpha
    # alpha=1  --> objeto totalmente opaco
    # alpha=0  --> opbjeto totalmente transparente
 
    glEnable(GL_DEPTH_TEST)
    glDepthFunc(GL_LEQUAL)
    #glDepthFunc(GL_LESS)
    
    # normaliza las normales luego del escalamiento.
    glEnable(GL_NORMALIZE)
	
def reshape((width, height)):
    if height == 0:
        height = 1
    glViewport(0, 0, width, height)
    glMatrixMode(GL_PROJECTION)
    glLoadIdentity()
    gluPerspective(60.0, float(width)/float(height), 0.1, 20000.0)
    #glOrtho(-w,w,-h,h,1,20000)
    glMatrixMode(GL_MODELVIEW)
    glLoadIdentity()

#####################################################################

def initLight():
    
	glLightfv(GL_LIGHT0, GL_POSITION, [ 3000.0, 0.0, 0.0, 1.0 ])
	glLightfv(GL_LIGHT0, GL_AMBIENT , [ 0.2, 0.2, 0.2, 1.0])
	glLightfv(GL_LIGHT0, GL_SPECULAR, [ 1.0, 1.0, 1.0, 1.0])
	glLightfv(GL_LIGHT0, GL_DIFFUSE , [ 1.0, 1.0, 1.0, 1.0])
 
	glLightf(GL_LIGHT0, GL_CONSTANT_ATTENUATION, 1.0)
	glLightf(GL_LIGHT0, GL_LINEAR_ATTENUATION, 0.0)
	glLightf(GL_LIGHT0, GL_QUADRATIC_ATTENUATION, 0.0)
 
	glLightf(GL_LIGHT0, GL_SPOT_CUTOFF, 180.0)
	glLightfv(GL_LIGHT0, GL_SPOT_DIRECTION, [ -1.0, 0.0, 0.0])
	glLightf(GL_LIGHT0, GL_SPOT_EXPONENT, 0.0)
 
	glEnable(GL_LIGHT0)

#####################################################################

def listaBotella(n):

    #iniciar lista
    lista = glGenLists(1)
    glNewList(lista,GL_COMPILE)
    
    dang = pi/float(n)
    
    # triang(d,a0,a1) - triang(d,a1,a2)
    # cuad(a1,a0,b0,b1) - cuad(a2,a1,b1,b2)
    # cuad(b1,b0,c0,c1) - cuad(b2,b1,c1,c2)
    # cuad(c1,c0,d0,d1) - cuad(c2,c1,d1,d2)
    # traing(d1,d0,e) - triang(d2,d1,e)
    
    d = v3d.Vector(0.0,0.0,1.0)
    e = v3d.Vector(0.0,0.0,0.0)
    
    nd = v3d.Vector(0.0,0.0,1.0)
    ne = v3d.Vector(0.0,0.0,-1.0)
    
    a0 = v3d.VectorCilindricas(0.2,0.0,1.0)
    b0 = v3d.VectorCilindricas(0.2,0.0,0.8)
    c0 = v3d.VectorCilindricas(0.5,0.0,0.6)
    d0 = v3d.VectorCilindricas(0.5,0.0,0.0)
    
    a1 = v3d.VectorCilindricas(0.2,dang,1.0)
    b1 = v3d.VectorCilindricas(0.2,dang,0.8)
    c1 = v3d.VectorCilindricas(0.5,dang,0.6)
    d1 = v3d.VectorCilindricas(0.5,dang,0.0)
    
    a2 = v3d.VectorCilindricas(0.2,2*dang,1.0)
    b2 = v3d.VectorCilindricas(0.2,2*dang,0.8)
    c2 = v3d.VectorCilindricas(0.5,2*dang,0.6)
    d2 = v3d.VectorCilindricas(0.5,2*dang,0.0)
    
    nA0 = v3d.normal(d,a0,a1)
    nB0 = v3d.normal(a1,a0,b0)
    nC0 = v3d.normal(b1,b0,c0)
    nD0 = v3d.normal(c1,c0,d0)
    nE0 = v3d.normal(d1,d0,e)
    
    nA1 = v3d.normal(d,a1,a2)
    nB1 = v3d.normal(a2,a1,b1)
    nC1 = v3d.normal(b2,b1,c1)
    nD1 = v3d.normal(c2,c1,d1)
    nE1 = v3d.normal(d2,d1,e)
    
    na0 = v3d.promediar([nA0,nA1,nB0,nB1])
    nb0 = v3d.promediar([nB0,nB1,nC0,nC1])
    nc0 = v3d.promediar([nC0,nC1,nD0,nD1])
    nd0 = v3d.promediar([nD0,nD1,nE0,nE1])
    
    ang = dang
    while ang<=2*pi+dang:
    
        a0 = v3d.VectorCilindricas(0.2,ang,1.0)
        b0 = v3d.VectorCilindricas(0.2,ang,0.8)
        c0 = v3d.VectorCilindricas(0.5,ang,0.6)
        d0 = v3d.VectorCilindricas(0.5,ang,0.0)
        
        a1 = v3d.VectorCilindricas(0.2,ang+dang,1.0)
        b1 = v3d.VectorCilindricas(0.2,ang+dang,0.8)
        c1 = v3d.VectorCilindricas(0.5,ang+dang,0.6)
        d1 = v3d.VectorCilindricas(0.5,ang+dang,0.0)
        
        a2 = v3d.VectorCilindricas(0.2,ang+2*dang,1.0)
        b2 = v3d.VectorCilindricas(0.2,ang+2*dang,0.8)
        c2 = v3d.VectorCilindricas(0.5,ang+2*dang,0.6)
        d2 = v3d.VectorCilindricas(0.5,ang+2*dang,0.0)
        
        nA0 = v3d.normal(d,a0,a1)
        nB0 = v3d.normal(a1,a0,b0)
        nC0 = v3d.normal(b1,b0,c0)
        nD0 = v3d.normal(c1,c0,d0)
        nE0 = v3d.normal(d1,d0,e)
        
        nA1 = v3d.normal(d,a1,a2)
        nB1 = v3d.normal(a2,a1,b1)
        nC1 = v3d.normal(b2,b1,c1)
        nD1 = v3d.normal(c2,c1,d1)
        nE1 = v3d.normal(d2,d1,e)
        
        na1 = v3d.promediar([nA0,nA1,nB0,nB1])
        nb1 = v3d.promediar([nB0,nB1,nC0,nC1])
        nc1 = v3d.promediar([nC0,nC1,nD0,nD1])
        nd1 = v3d.promediar([nD0,nD1,nE0,nE1])
        
        # 'v' indica que argumentos son del tipo Vector3D
        vdrawTrianglen3(d,a0,a1,nd,na0,na1)
        vdrawQuadn4(a1,a0,b0,b1,na1,na0,nb0,nb1)
        vdrawQuadn4(b1,b0,c0,c1,nb1,nb0,nc0,nc1)
        vdrawQuadn4(c1,c0,d0,d1,nc1,nc0,nd0,nd1)
        vdrawTrianglen3(d1,d0,e,nd1,nd0,ne)
        
        na0 = na1
        nb0 = nb1
        nc0 = nc1
        nd0 = nd1
        
        ang += dang
    
    #terminar lista
    glEndList()
    
    return lista

#####################################################################

W=640 #ancho de ventana
H=480 #alto de ventana

# inicializando ...
init_pygame((W,H),"botella")
init_opengl((W,H))

# imprime información sobre el Hardware gráfico y
# la version de OpenGL implementada
printVersions()

# creación de dibujos en listas
axes=axesList(1000)
bot = listaBotella(5)

# variables del programa
o = 30
w = 0.05
vzoom = 0.1
eye = 200.0

show_axes = True
fill_polygons = True
smooth_flat = True
zoom_plus = False
zoom_minus = False

# configuración de camara
glLoadIdentity()
gluLookAt( 2000.0, 2000.0, 1000.0, \
          0.0, 0.0, 0.0, \
          0.0, 0.0, 1.0)
          
# habilita iluminación
glEnable(GL_LIGHTING)

# configura fuentes de luz
initLight()

# medida de tiempo inicial
t0 = pygame.time.get_ticks()

run = True
while run:
    # 0: CONTROL DEL TIEMPO
    t1 = pygame.time.get_ticks()    # tiempo actual
    dt = (t1 - t0)                  # diferencial de tiempo asociado a la iteración
    t0 = t1                         # actualizar tiempo inicial para siguiente iteración
    
    # 1: MANEJAMOS EVENTOS DE ENTRADA (TECLADO, MOUSE, ETC.)
    for event in pygame.event.get():
        if event.type == QUIT:
            run = False
        if event.type == KEYDOWN:
            if event.key == K_ESCAPE:
                run = False
            if event.key == K_1:
                show_axes = not show_axes
            if event.key == K_2:
                fill_polygons = not fill_polygons
            if event.key == K_3:
                smooth_flat = not smooth_flat
                if smooth_flat:
                    print "smooth"
                else:
                    print "flat"
            if event.key == K_UP:
                zoom_plus = True
            if event.key == K_DOWN:
                zoom_minus = True
        if event.type == KEYUP:
            if event.key == K_UP:
                zoom_plus = False
            if event.key == K_DOWN:
                zoom_minus = False
				
    # 2: EJECUTAMOS LOGICA DE LA APLICACION    
    o += w*dt
    
    if zoom_plus:
        eye += vzoom*dt
        
    if zoom_minus:
        eye -= vzoom*dt
        if (eye < 0.0):
            eye = 0.0
        
    glLoadIdentity()
    gluLookAt( eye, eye, eye, \
              0.0, 0.0, 0.0, \
              0.0, 0.0, 1.0)
	
    # 3: DIBUJAMOS LOS ELEMENTOS
    # limpia la pantalla (buffer de color y de profundidad)
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT) 
    
    #posibilidad de pintar o no los polígonos
    if fill_polygons:
        glPolygonMode(GL_FRONT, GL_FILL)
        glPolygonMode(GL_BACK, GL_FILL)
    else:
        glPolygonMode(GL_FRONT, GL_LINE)
        glPolygonMode(GL_BACK, GL_LINE)
    
    # posibilidad de mostrar o no los ejes
    if show_axes:
        glDisable(GL_LIGHTING)
        glCallList(axes)
        glEnable(GL_LIGHTING)
    
    if smooth_flat:
        glShadeModel(GL_SMOOTH)
    else:
        glShadeModel(GL_FLAT)
    
    glMaterialfv(GL_FRONT, GL_AMBIENT, [0.25,0.5,0.25,1.0])
    glMaterialfv(GL_FRONT, GL_DIFFUSE, [0.5,1.0,0.5,1.0])
    glMaterialfv(GL_FRONT, GL_SPECULAR,[1.0,1.0,1.0,1.0])
    glMaterialfv(GL_FRONT, GL_SHININESS, [10.0])
    glMaterialfv(GL_FRONT, GL_EMISSION, [0.0,0.0,0.0,1.0])
    
    drawList(bot,o = o,rot = [0,0,1],sz = [100,60,150])
    
    pygame.display.flip()       # lanza el dibujo a la pantalla
    pygame.time.wait(1000/30)   # ajusta para trabajar a 30 fps.

#####################################################################
